#ifndef __F_H_INCLUDED_
#define __F_H_INCLUDED_

#include <Eigen/LU>
using namespace Eigen;

// rhs of ode
template <typename T, int N=Dynamic>
inline void g(const Matrix<T,3,1>& p, const Matrix<T,N,1>& y, 
              Matrix<T,N,1>& r) {
  int n=y.size();
  for (int i=0;i<n;i++) {
    r(i)=p(0)*(n+1)*(n+1);
    if (i==0)
      r(i)*=p(1)-2*y(i)+y(i+1);
    else if (i==n-1)
      r(i)*=y(i-1)-2*y(i)+p(2);
    else 
      r(i)*=y(i-1)-2*y(i)+y(i+1);
  }
}

// tangent of rhs of ode
template <typename T, int N=Dynamic>
inline void g_t(const Matrix<T,3,1>& p, const Matrix<T,N,1>& y, 
                const Matrix<T,N,1>& y_t, Matrix<T,N,1>& r_t) {
  int n=y.size();
  for (int i=0;i<n;i++) {
      r_t(i)=p(0)*(n+1)*(n+1);
    if (i==0) 
      r_t(i)*=-2*y_t(i)+y_t(i+1);
    else if (i==n-1) 
      r_t(i)*=y_t(i-1)-2*y_t(i);
    else 
      r_t(i)*=y_t(i-1)-2*y_t(i)+y_t(i+1);
  }
}

// Jacobian of rhs of ode
template <typename T, int N=Dynamic>
inline void dfdy(const int m, const Matrix<T,3,1>& p, 
                 const Matrix<T,N,1>& y, Matrix<T,N,N>& A) {
  int n=y.size();
  Matrix<T,N,1> r=Matrix<T,N,1>::Zero(n), r_t=Matrix<T,N,1>::Zero(n);
  for (int i=0;i<n;i++) {
    Matrix<T,N,1> y_t=Matrix<T,N,1>::Zero(n);
    y_t(i)=1;
    g_t(p,y,y_t,r_t);
    if (i>0) A(i,i-1)=-r_t(i-1)/m;
    A(i,i)=-r_t(i)/m+1;
    if (i<n-1) A(i,i+1)=-r_t(i+1)/m;
  }
}

// residual of nls
template <typename T, int N=Dynamic>
inline void f(const int m, const Matrix<T,3,1>& p, 
              const Matrix<T,N,1>& y, const Matrix<T,N,1>& y_prev, 
              Matrix<T,N,1>& r) {
  g(p,y,r); r=y-y_prev-r/m;
}

// Newton solver for nls
template <typename T, int N=Dynamic>
inline void newton(const int m, const Matrix<T,3,1>& p, 
                   const Matrix<T,N,1>& y_prev, Matrix<T,N,1>& y) {
  int n=y.size();
  const double eps=1e-14;
  Matrix<T,N,N> A=Matrix<T,N,N>::Zero(n,n); 
  Matrix<T,N,1> r=Matrix<T,N,1>::Zero(n);
  f(m,p,y,y_prev,r); 
  while (r.norm()>eps) {
    dfdy(m,p,y,A);
    PartialPivLU<Matrix<T,N,N>> LU(A);
    y-=LU.solve(r); 
    f(m,p,y,y_prev,r); 
  }
}

// implicit Euler integration
template <typename T, int N=Dynamic>
inline void f(const int m, const Matrix<T,3,1>& p, Matrix<T,N,1>& y) {
  for (int j=0;j<m;j++) {
    Matrix<T,N,1> y_prev=y;
    newton(m,p,y_prev,y); 
  }
}

#endif
